/******************************************************************************
* File    : pmc_util.c
*
* Purpose : Various PMC functions to support PMC testing. 
* Author  : Abhisek Verma, Freescale Semiconductor
* Email   : b05791@freescale.com
* Date    : 12.06.2007
******************************************************************************
* Notes:
* 
*****************************************************************************/
# include "intrinsics_56800E.h"

/*   Global variables section      */
// Global variable can be referenced from the main and isr files by using extern qualifier
uint8_t flag_isr;


/*   Global defines section        */

# define RUN    (uint8_t)0x00
# define LPRUN  (uint8_t)0x01
# define WAIT   (uint8_t)0x02
# define LPWAIT (uint8_t)0x03
# define STOP   (uint8_t)0x04
# define LPSTOP (uint8_t)0x05
# define PPDM   (uint8_t)0x0C
# define MB_MODE_CHECK   255


/*USE OF MACROS TO ENTER_EXIT LPRUN MODE*/
// to set LPR and clear LVDE at the same time
// keep LVDRE enabled
# define ENTER_LPRUN W16(PMC_SCR,0x0280)

// clr LPR bit to come out of LPRUN mode
// this operation will also clr the porf bit 
// if called for the first time.
# define EXIT_LPRUN  BCLR16(PMC_SCR,PMC_SCR_LPR)



/*   Function prototype section    */

void check_mode(uint8_t);
void enable_hawk_interrupts(void);
void enter_wait_mode(void);
void gen_interrupt_gpioa0(void);
void configure_gpioa0(void);
void gpioa0_isr_done(void);
void enter_stop_mode(void);
void initialize_at_por(uint32_t);
void enter_lpwait_mode(void);
void enter_lpstop_mode(void);
void enter_ppd_mode(void);
void configure_cmp0(void);

/*   Function definition section    */
/******************************************************************************
* check_mode(MODE) 
* This function checks the current mode of the SOC as per the defines above.
* Its correct implementation is in the verilog task.
* 
*****************************************************************************/


void check_mode(uint8_t mode)
{  
        INFO("SW:pmc_util():","Checking current user mode");
	MB_PUT8(MB_MODE_CHECK,mode);
}

/******************************************************************************
* enable_hawk_interrupts() 
* This function enables hawk interrupts.
* 
*****************************************************************************/
void enable_hawk_interrupts() {
   INFO("SW:pmc_util():", "Enable enable_hawk_interrupts");
   asm(moveu.w #$0000,sr);
}   

/******************************************************************************
* enter_wait_mode() 
* This function puts the cpu in wait state.
* 
*****************************************************************************/
void enter_wait_mode(void)
{
	INFO("SW:pmc_util():","Clear LPR bit in PMC_SCR");
	BCLR16(PMC_SCR,PMC_SCR_LPR);
	REM16(PMC_SCR,0x0000,PMC_SCR_LPR|PMC_SCR_LPRS);
	//check LVDE is enabled
	REM16(PMC_SCR,0x0008,PMC_SCR_LVDE);
	REM16(SIM_CTRL,0x0000,SIM_CTRL_WAIT_disable_0);
	INFO("SW:pmc_util():","Calling intrinsic wait function");
	wait();
}

/******************************************************************************
* gen_interrupt_gpioa0() 
* This function generates h/w interrupt on gpioa0 pin from the ext world.
* It also checks that its isr is serviced.This function Configures GPIOA[0] for 
* active high edge sensitive interrupt
* 
*****************************************************************************/
void gen_interrupt_gpioa0(void)
{
   configure_gpioa0();
   
   INFO("SW:pmc_util():","Generate a h/w interrupt at GPIOA[0]");
   // drive one on GPIOA[0] for an interrupt to occur.
   MB_PUT16(SET_GPIO,PIN_GPIOA0|DRV1);

   gpioa0_isr_done();
}

/******************************************************************************
* configure_gpioa0() 
* This function Configures GPIOA[0] for 
* active high edge sensitive interrupt on gpioa0 pin from the ext world.
*  
*****************************************************************************/
void configure_gpioa0(void)
{
   INFO("SW:pmc_util():","Set flag_isr to zero");
   flag_isr=0x00;
   // first drive zero on GPIOA[0].
   MB_PUT16(SET_GPIO,PIN_GPIOA0|DRV0);
   // do the register settings for edge and pol.
   W16(GPIOA_IENR,0x0001);
   RE16(GPIOA_IENR,0x0001);
   W16(GPIOA_IPOLR,0x0000);
   RE16(GPIOA_IPOLR,0x0000);
}

/******************************************************************************
* gpioa0_isr_done() 
* This function checks that the isr for gpioa0 int. is serviced.
* *****************************************************************************/
void gpioa0_isr_done(void)
{
   WAIT_RE8(flag_isr,0x01,20); 
   if (flag_isr==1)
   {	INFO ("SW:pmc_util():","Back from ISR");  }
   else
   {	ERROR ("SW:pmc_util():","ISR didnt execute"); }
}
	

/******************************************************************************
* enter_stop_mode() 
* This function puts the cpu in stop state.
* 
*****************************************************************************/
void enter_stop_mode(void)
{
	INFO("SW:pmc_util():","Clear LPR bit in PMC_SCR");
	BCLR16(PMC_SCR,PMC_SCR_LPR);
	REM16(PMC_SCR,0x0000,PMC_SCR_LPR|PMC_SCR_LPRS);
	//check LVDE is set
	REM16(PMC_SCR,0x0008,PMC_SCR_LVDE);
	REM16(SIM_CTRL,0x0000,SIM_CTRL_STOP_disable_0);
	INFO("SW:pmc_util():","Calling intrinsic stop function");
	stop();
}

/******************************************************************************
* initialize_at_por() 
* This function initializes the variables passed only after por reset.
* 
*****************************************************************************/

void initialize_at_por(uint32_t var)
{
	uint16_t sim_rstat_val;
	// Read SIM reset status reg after POR.
	// If it was POR initialize the reset flags.
	R16(SIM_RSTAT,sim_rstat_val);
   if (sim_rstat_val & SIM_RSTAT_POR)
	{
		INFO("SW:pmc_util():", " First reset occurred---POR");
		
		RE16(PMC_SCR,PMC_SCR_PORF|PMC_SCR_LVDRE|PMC_SCR_LVDE);
		
		WR16(MEM_PTR16(var),0x0000);
	}
}
/******************************************************************************
* enter_lpwait_mode() 
* This function puts the soc in lpwait state.
* 
*****************************************************************************/
void enter_lpwait_mode(void)
{
	
	INFO("SW:pmc_util():","Clear LVDE bit in PMC_SCR");
	BCLR16(PMC_SCR,PMC_SCR_LVDE);
	REM16(PMC_SCR,0x0000,PMC_SCR_LVDE);
	INFO("SW:pmc_util():","SET LPR bit in PMC_SCR");
	BSET16(PMC_SCR,PMC_SCR_LPR);
	INFO("SW:pmc_util():","Wait for LPRS bit to get set in PMC_SCR");
	WAIT_REM16(PMC_SCR,0x00C0,PMC_SCR_LPR|PMC_SCR_LPRS,0x20);
	check_mode(LPRUN);
	REM16(SIM_CTRL,0x0000,SIM_CTRL_WAIT_disable_0);
	INFO("SW:pmc_util():","Calling intrinsic wait function");
	wait();
}

/******************************************************************************
* enter_lpstop_mode() 
* This function puts the soc in lpstop state.
* 
*****************************************************************************/
void enter_lpstop_mode(void)
{
	
	INFO("SW:pmc_util():","Clear LVDE bit in PMC_SCR");
	BCLR16(PMC_SCR,PMC_SCR_LVDE);
	REM16(PMC_SCR,0x0000,PMC_SCR_LVDE);
	INFO("SW:pmc_util():","SET LPR bit in PMC_SCR");
	BSET16(PMC_SCR,PMC_SCR_LPR);
	INFO("SW:pmc_util():","Wait for LPRS bit to get set in PMC_SCR");
	WAIT_REM16(PMC_SCR,0x00C0,PMC_SCR_LPR|PMC_SCR_LPRS,0x20);
	check_mode(LPRUN);
	REM16(SIM_CTRL,0x0000,SIM_CTRL_STOP_disable_0);
	INFO("SW:pmc_util():","Calling intrinsic stop function");
	stop();
}

/******************************************************************************
* enter_ppd_mode() 
* This function puts the soc in PPD state.
* 
*****************************************************************************/
void enter_ppd_mode(void)
{
	
	INFO("SW:pmc_util():","Clear LVDE bit in PMC_SCR");
	BCLR16(PMC_SCR,PMC_SCR_LVDE);
	REM16(PMC_SCR,0x0000,PMC_SCR_LVDE);
	INFO("SW:pmc_util():","SET PPDE bit in PMC_SCR");
	BSET16(PMC_SCR,PMC_SCR_PPDE);
	REM16(PMC_SCR,0x0100,PMC_SCR_PPDE);
	REM16(SIM_CTRL,0x0000,SIM_CTRL_STOP_disable_0);
	INFO("SW:pmc_util():","Calling intrinsic stop function");
	stop();
}

/******************************************************************************
* configure_cmp0() 
* This function configure cmp0 for interrupt generation.
* 
*****************************************************************************/

void configure_cmp0(void)
{
	//initialize pins GPIO-E1,E2 for CMP-0
	//the deal you cant keep the inputs at the same level
	MB_PUT16(SET_GPIO,PIN_GPIOE1|DRV0);
	MB_PUT16(SET_GPIO,PIN_GPIOE2|DRV1);

	//enable the comparator
	//ope needs to be set--bug 6247
	BSET16(CMP0_CR1,CMP0_CR1_EN);
	RE16(CMP0_CR1,0x0001);
	//configure for rising edge of compare
	BSET16(CMP0_SCR,0x0010);
	RE16(CMP0_SCR,0x0010);
}

